from django.contrib.auth import authenticate
from django.contrib.auth.models import User
from django.utils.translation import ugettext_lazy as _
from rest_framework import serializers

from App.loyaltyMe.serializers import PromotionSerializer, TaskSerializer
from App.user.models import UserProfile, UserPromotion, UserTask


class UserSerializer(serializers.HyperlinkedModelSerializer):
    class Meta:
        model = User
        fields = ('url', 'id', 'username', 'email',)


class UserProfileSerializer(serializers.HyperlinkedModelSerializer):
    user = UserSerializer()

    class Meta:
        model = UserProfile
        fields = ('url', 'id', 'user', 'photo')



class UserTaskSerializer(serializers.HyperlinkedModelSerializer):
    user = UserSerializer()
    task = TaskSerializer()
    last_update = serializers.DateTimeField(format="%Y-%m-%dT%H:%M:%S")
    beacon_signal_last_update = serializers.DateTimeField(format="%Y-%m-%dT%H:%M:%S")

    class Meta:
        model = UserTask
        fields = ('url', 'id', 'task', 'user', 'done', 'last_update', 'counter', 'beacon_counter', 'beacon_signal_last_update')


class AuthTokenSerializerByEmail(serializers.Serializer):
    email = serializers.CharField(label=_("Email"))
    password = serializers.CharField(label=_("Password"), style={'input_type': 'password'})

    def validate(self, attrs):
        email = attrs.get('email')
        password = attrs.get('password')
        user_query = User.objects.filter(email__iexact=email)
        if not user_query:
            msg = _('Unable to log in with provided credentials.')
            raise serializers.ValidationError(msg)
        user_by_email = user_query[0]
        if user_by_email and password:
            user = authenticate(username=user_by_email.username, password=password)

            if user:
                if not user.is_active:
                    msg = _('User account is disabled.')
                    raise serializers.ValidationError(msg)
            else:
                msg = _('Unable to log in with provided credentials.')
                raise serializers.ValidationError(msg)
        else:
            msg = _('Must include "username" and "password".')
            raise serializers.ValidationError(msg)

        attrs['user'] = user
        return attrs


class UserPromotionSerializer(serializers.HyperlinkedModelSerializer):
    user = UserSerializer()
    promotion = PromotionSerializer()

    class Meta:
        model = UserPromotion
        fields = ('url', 'id', 'user', 'place', 'promotion', 'progress', 'code')