from django.contrib.auth import login, logout
from django.contrib.auth.decorators import login_required
from django.core.urlresolvers import reverse
from django.shortcuts import render, redirect

from django.utils.decorators import method_decorator
from django.views import generic
from django.views.decorators.csrf import csrf_exempt
from django.views.generic import FormView, View, DetailView, ListView, CreateView, UpdateView
from django.views.generic.base import TemplateResponseMixin

from App.loyaltyMe.models import UserPlace, Place
from App.page.forms import EmailAuthenticationForm


class CommingSoonView(generic.TemplateView):
    template_name = 'page/base.html'


class MainPageView(generic.TemplateView):
    template_name = 'main_page/base.html'


class PlaceCreateView(CreateView):
    template_name = 'panel/place_form.html'
    model = Place
    fields = ['name', 'type', 'description']

    def get_success_url(self):
        return reverse('place_detail', args=(self.object.id,))

    def form_valid(self, form):
        self.object = form.save()
        UserPlace.objects.create(user=self.request.user, place=self.object)
        return super(PlaceCreateView, self).form_valid(form)


class PlaceUpdateView(UpdateView):
    template_name = 'panel/place_form.html'
    model = Place
    fields = ['name', 'type', 'description']

    def get_success_url(self):
        return reverse('place_detail', args=(self.object.id,))


class PlacePageListView(ListView):
    template_name = 'panel/place_list.html'
    model = Place

    def get_queryset(self):
        qs = super(PlacePageListView, self).get_queryset()
        return qs.filter(userplace_place__user=self.request.user)

    def get_context_data(self, **kwargs):
        context = super(PlacePageListView, self).get_context_data(**kwargs)
        return context


class PlacePageDetailView(DetailView):
    template_name = 'panel/place_detail.html'
    model = Place

    def get_queryset(self):
        qs = super(PlacePageDetailView, self).get_queryset()
        return qs.filter(userplace_place__user=self.request.user)

    def get_context_data(self, **kwargs):
        context = super(PlacePageDetailView, self).get_context_data(**kwargs)
        return context


@method_decorator(login_required, name='dispatch')
class PanelPageView(generic.TemplateView):
    template_name = 'panel/panel.html'

    def get(self, request, *args, **kwargs):
        if request.user.is_authenticated():
            return render(request, self.template_name, {'user': request.user})
        return render(request, self.template_name)


@method_decorator(csrf_exempt, name='dispatch')
class LoginView(FormView):
    template_name = 'panel/login.html'
    form_class = EmailAuthenticationForm
    success_url = '/panel'

    def form_valid(self, form):
        login(self.request, form.get_user())
        return super(LoginView, self).form_valid(form)

    def get(self, request, *args, **kwargs):
        if request.user.is_authenticated():
            return redirect('main_page')
        return render(request, self.template_name, {'form': self.form_class})


class LogoutView(TemplateResponseMixin, View):
    template_name = 'panel/logout.html'

    def get(self, request, *args, **kwargs):
        if request.user.is_authenticated():
            logout(self.request)
            return render(request, self.template_name)
        return render(request, self.template_name)
