from django.contrib.auth import login, logout
from django.contrib.auth.decorators import login_required
from django.contrib.auth.models import User
from django.core.urlresolvers import reverse, reverse_lazy
from django.http import HttpResponseRedirect
from django.shortcuts import render, redirect
from django.utils.decorators import method_decorator
from django.views import generic
from django.views.decorators.csrf import csrf_exempt
from django.views.generic import FormView, View, DetailView, ListView, CreateView, UpdateView, DeleteView
from django.views.generic.base import TemplateResponseMixin

from App.loyaltyMe.models import Place, Beacon
from App.page.forms import EmailAuthenticationForm, PlaceForm, PromotionForm
from App.promotion.models import PromotionSet, Promotion
from App.user.models import UserProfile


class CommingSoonView(generic.TemplateView):
    template_name = 'page/base.html'


class MainPageView(generic.TemplateView):
    template_name = 'main_page/base.html'


class PlacePageListView(ListView):
    template_name = 'panel/place_list.html'
    model = Place

    def get_queryset(self):
        qs = super(PlacePageListView, self).get_queryset()
        return qs.filter(owners=self.request.user)

    def get_context_data(self, **kwargs):
        context = super(PlacePageListView, self).get_context_data(**kwargs)
        return context


class PlacePageDetailView(DetailView):
    template_name = 'panel/place_detail.html'
    model = Place

    def get_queryset(self):
        qs = super(PlacePageDetailView, self).get_queryset()
        return qs.filter(owners=self.request.user)

    def get_context_data(self, **kwargs):
        context = super(PlacePageDetailView, self).get_context_data(**kwargs)
        return context


class PlaceCreateView(CreateView):
    template_name = 'panel/place_form.html'
    form_class = PlaceForm

    def get_success_url(self, id):
        return reverse('place_detail', args=(id,))

    def get_form_kwargs(self):
        kwargs = super(PlaceCreateView, self).get_form_kwargs()
        kwargs['user'] = self.request.user
        return kwargs

    def form_valid(self, form):
        new_place = form.save()
        new_place.owners.add(self.request.user)
        new_place.save()
        return HttpResponseRedirect(self.get_success_url(new_place.id))


class PlaceUpdateView(UpdateView):
    template_name = 'panel/place_form.html'
    form_class = PlaceForm

    def get_success_url(self):
        return reverse('place_detail', args=(self.object.id,))

    def get_form_kwargs(self):
        kwargs = super(PlaceCreateView, self).get_form_kwargs()
        kwargs['user'] = self.request.user
        return kwargs


class BeaconPageListView(ListView):
    template_name = 'panel/beacon_list.html'
    model = Beacon

    def get_queryset(self):
        qs = super(BeaconPageListView, self).get_queryset()
        return qs.filter(user=self.request.user)

    def get_context_data(self, **kwargs):
        context = super(BeaconPageListView, self).get_context_data(**kwargs)
        return context


class BeaconPageDetailView(DetailView):
    template_name = 'panel/beacon_detail.html'
    model = Beacon

    def get_queryset(self):
        qs = super(BeaconPageDetailView, self).get_queryset()
        return qs.filter(user=self.request.user)

    def get_context_data(self, **kwargs):
        context = super(BeaconPageDetailView, self).get_context_data(**kwargs)
        return context


class BeaconCreateView(CreateView):
    template_name = 'panel/beacon_form.html'
    model = Beacon
    fields = ['name', 'description', 'uuid', 'major', 'minor']

    def get_success_url(self, id):
        return reverse('place_detail', args=(id,))


    def form_valid(self, form):
        new_beacon = form.save(commit=False)
        new_beacon.user = self.request.user
        new_beacon.save()
        return HttpResponseRedirect(self.get_success_url(new_beacon.id))


class BeaconUpdateView(UpdateView):
    template_name = 'panel/beacon_form.html'
    model = Beacon
    fields = ['name', 'description', 'uuid', 'major', 'minor']

    def get_success_url(self):
        return reverse('beacon_detail', args=(self.object.id,))


@method_decorator(login_required, name='dispatch')
class PanelPageView(generic.TemplateView):
    template_name = 'panel/panel.html'

    def get(self, request, *args, **kwargs):
        if request.user.is_authenticated():
            return render(request, self.template_name, {'user': request.user})
        return render(request, self.template_name)


@method_decorator(csrf_exempt, name='dispatch')
class LoginView(FormView):
    template_name = 'panel/login.html'
    form_class = EmailAuthenticationForm
    success_url = '/panel'

    def form_valid(self, form):
        login(self.request, form.get_user())
        return super(LoginView, self).form_valid(form)

    def get(self, request, *args, **kwargs):
        if request.user.is_authenticated():
            return redirect('main_page')
        return render(request, self.template_name, {'form': self.form_class})


class LogoutView(TemplateResponseMixin, View):
    template_name = 'panel/logout.html'

    def get(self, request, *args, **kwargs):
        if request.user.is_authenticated():
            logout(self.request)
            return render(request, self.template_name)
        return render(request, self.template_name)


class UserPageDetailView(DetailView):
    template_name = 'panel/user_detail.html'
    model = UserProfile

    def get_context_data(self, **kwargs):
        context = super(UserPageDetailView, self).get_context_data(**kwargs)
        return context


class PromotionSetPageDetailView(DetailView):
    template_name = 'panel/promotion_set_detail.html'
    model = PromotionSet

    def get_context_data(self, **kwargs):
        context = super(PromotionSetPageDetailView, self).get_context_data(**kwargs)
        return context


class PromotionSetCreateView(CreateView):
    template_name = 'panel/promotion_set_form.html'
    model = PromotionSet
    fields = ['name', ]

    def get_context_data(self, **kwargs):
        context = super(PromotionSetCreateView, self).get_context_data(**kwargs)

        return context

    def get_success_url(self, id):
        return reverse('promotion_set_detail', args=(id,))

    def form_valid(self, form, **kwargs):
        new_promotion_set = form.save()
        place = Place.objects.get(pk=self.kwargs.get('pk'))
        new_promotion_set.place_promotion.add(place)
        new_promotion_set.save()
        return HttpResponseRedirect(self.get_success_url(new_promotion_set.id))


class PromotionPageDetailView(DetailView):
    template_name = 'panel/promotion_detail.html'
    model = Promotion

    def get_context_data(self, **kwargs):
        context = super(PromotionPageDetailView, self).get_context_data(**kwargs)
        return context



class PromotionCreateView(CreateView):
    template_name = 'panel/promotion_form.html'
    model = Promotion
    form_class = PromotionForm

    def get_context_data(self, **kwargs):
        context = super(PromotionCreateView, self).get_context_data(**kwargs)

        return context

    def get_success_url(self, id):
        return reverse('promotion_detail', args=(id,))

    def form_valid(self, form, **kwargs):
        new_promotion = form.save()
        promotion_set = PromotionSet.objects.get(pk=self.kwargs.get('pk'))
        promotion_set.promotions.add(new_promotion)
        new_promotion.save()
        return HttpResponseRedirect(self.get_success_url(new_promotion.id))

class PromotionDeleteView(DeleteView):
    template_name = 'panel/promotion_confirm_delete.html'
    model = Promotion
    success_url = reverse_lazy('place_list')