import django_filters
import rest_framework
from django.contrib.auth.models import User

# Create your views here.
from rest_framework import filters
from rest_framework.authtoken.models import Token
from rest_framework.authtoken.serializers import AuthTokenSerializer
from rest_framework.decorators import list_route, api_view, renderer_classes
from rest_framework.response import Response
from rest_framework import viewsets, permissions
from rest_framework.views import APIView
from rest_framework_swagger.renderers import OpenAPIRenderer, SwaggerUIRenderer
from rest_framework import response, schemas

from App.user.models import UserProfile, UserPromotion, UserTask
from App.user.serializers import UserSerializer, AuthTokenSerializerByEmail, UserProfileSerializer, \
    UserPromotionSerializer, UserTaskSerializer
from rest_framework import parsers, renderers


class UserViewSet(viewsets.ModelViewSet):
    """
    API for articles
    """
    queryset = User.objects.all()
    serializer_class = UserSerializer
    permission_classes = (permissions.IsAuthenticatedOrReadOnly,)

    @list_route(methods=['get'])
    def me(self, request, pk=None):
        user = User.objects.filter(pk=request.user.id)[0]
        user_data = UserSerializer(user, context={'request': request}).data
        return Response(status=rest_framework.status.HTTP_200_OK, data=user_data)


class UserProfileViewSet(viewsets.ModelViewSet):
    """
    Returns a list of all **active** accounts in the system.

    For more details on how accounts are activated please [see here][ref].

    [ref]: http://example.com/activating-accounts
    """
    queryset = UserProfile.objects.all()
    serializer_class = UserProfileSerializer
    permission_classes = (permissions.IsAuthenticatedOrReadOnly,)

    def perform_create(self, serializer):
        serializer.save(user=self.request.user)

    def list(self, request):
        user = request.user
        queryset = UserProfile.objects.exclude(user=user)
        page = self.paginate_queryset(queryset)
        serializer = UserProfileSerializer(page, many=True, context={'request': request})
        return self.get_paginated_response(serializer.data)

    @list_route(methods=['get'])
    def me(self, request, pk=None):
        user = request.user
        user_profile = UserProfile.objects.filter(user=user)[0]
        user_profile_data = UserProfileSerializer(user_profile, context={'request': request}).data
        return Response(status=rest_framework.status.HTTP_200_OK, data=user_profile_data)


class ObtainAuthToken(APIView):
    throttle_classes = ()
    permission_classes = ()
    parser_classes = (parsers.FormParser, parsers.MultiPartParser, parsers.JSONParser,)
    renderer_classes = (renderers.JSONRenderer,)
    serializer_class = AuthTokenSerializerByEmail

    def post(self, request, *args, **kwargs):
        serializer = self.serializer_class(data=request.data)
        serializer.is_valid(raise_exception=True)
        user = serializer.validated_data['user']
        token, created = Token.objects.get_or_create(user=user)
        return Response({'token': token.key})

    def get_serializer_class(self):
        return AuthTokenSerializerByEmail


obtain_auth_token = ObtainAuthToken.as_view()


class UserPromotionFilter(filters.FilterSet):
    place = django_filters.CharFilter(name="place__id")

    class Meta:
        model = UserPromotion
        fields = ['place']


class UserPromotionViewSet(viewsets.ModelViewSet):
    """
    API for articles
    """
    queryset = UserPromotion.objects.all()
    serializer_class = UserPromotionSerializer
    permission_classes = (permissions.IsAuthenticatedOrReadOnly,)
    filter_backends = (filters.DjangoFilterBackend,)
    filter_class = UserPromotionFilter

    def get_queryset(self):
        """
        This view should return a list of all the purchases
        for the currently authenticated user.
        """
        user = self.request.user
        return UserPromotion.objects.filter(user=user)

class UserTaskViewSet(viewsets.ModelViewSet):
    """
    API for articles
    """
    queryset = UserTask.objects.all()
    serializer_class = UserTaskSerializer
    permission_classes = (permissions.IsAuthenticatedOrReadOnly,)
    filter_backends = (filters.DjangoFilterBackend,)

    def get_queryset(self):
        """
        This view should return a list of all the purchases
        for the currently authenticated user.
        """
        user = self.request.user
        return UserTask.objects.filter(user=user)


@api_view()
@renderer_classes([OpenAPIRenderer, SwaggerUIRenderer])
def schema_view(request):
    generator = schemas.SchemaGenerator(title='Bookings API')
    return response.Response(generator.get_schema(request=request))
