# -*- coding: utf-8 -*-
import requests
import time
from tornado.ioloop import IOLoop
from functools import partial
from db import DDBstations, DDBtypes
from dateutil.parser import parse
import models
from tornado.gen import coroutine


def get_station_by_id(station_id, station_data):
    for station in station_data:
        if str(station['api_station_id']) == station_id:
            return models.Station.get(station['station_id'])


def get_type_by_name(type_name, type_data):
    for type in type_data:
        if type['shortname'] == type_name:
            return models.Type.get(type['type_id'])


def get_timestamp_from_datetime(datetime_string):
    dt = parse(datetime_string)
    return time.mktime(dt.timetuple())


@coroutine
def main(ioloop):
    stations = DDBstations()
    types = DDBtypes()

    apiURL = "http://powietrze.malopolska.pl/_powietrzeapi/api/dane?act=danemiasta&ci_id=1"

    r = requests.get(apiURL)
    actual_data = r.json()['dane']['actual']

    station_data = ioloop.run_sync(partial(stations.get_all))
    type_data = ioloop.run_sync(partial(types.get_all))

    for station_measurements in actual_data:
        for measurement in station_measurements['details']:
            ready_station = get_station_by_id(measurement['st_id'], station_data)
            ready_type = get_type_by_name(measurement['par_name'], type_data)
            ready_measurement_value = measurement['o_value']
            ready_measurement_timestamp = get_timestamp_from_datetime(measurement['o_czas'])

            if ready_station and ready_type and ready_measurement_value and ready_measurement_timestamp:
                measurement = models.Measurement(station=ready_station, pollution_type=ready_type,
                                                 value=ready_measurement_value, time=ready_measurement_timestamp)
                yield measurement.save()

ioloop = IOLoop.instance()
IOLoop.current().run_sync(main(ioloop))
