import uuid

from db import DDBtypes, DDBmeasurements, DDBstations, DDBlastmeasurements
from functools import partial
from tornado.gen import coroutine

from tornado.ioloop import IOLoop


class DDBobject(object):
    @classmethod
    def from_dict(cls, data):
        return cls(**data)


class Station(DDBobject):
    stored = {}

    def __init__(self, api_station_id, city, longitude, latitude, name, station_id=None):
        self.id = station_id or str(uuid.uuid4())
        self.api_id = api_station_id
        self.city = city
        self.longitude = longitude
        self.latitude = latitude
        self.name = name

    @classmethod
    @coroutine
    def get(cls, station_id):
        if cls.stored.get(station_id, None):
            return cls.stored.get(station_id)

        result = yield DDBstations().get(station_id)
        return cls.from_dict(result)

    @coroutine
    def save(self):
        result = yield DDBstations().add(station=self)
        return result

    @classmethod
    @coroutine
    def get_all(cls):
        if not cls.stored:
            data = yield DDBstations().get_all()
            return_value = []
            for station in data:
                return_value.append(cls.from_dict(station))
            cls.stored = {i.id: i for i in return_value}

        return cls.stored


class Type(DDBobject):
    stored = {}

    def __init__(self, shortname, unit, norm, longname, description, type_id=None):
        self.id = type_id or str(uuid.uuid4())
        self.shortname = shortname
        self.unit = unit
        self.norm = norm
        self.longname = longname
        self.description = description

    @classmethod
    @coroutine
    def get(cls, type_id):
        if cls.stored.get(type_id, None):
            return cls.stored.get(type_id)

        result = yield DDBtypes().get(type_id)
        return cls.from_dict(result)

    @coroutine
    def save(self):
        result = yield DDBtypes().add(pollution_type=self)
        return result

    @classmethod
    @coroutine
    def get_all(cls):
        if not cls.stored:
            data = yield DDBtypes().get_all()
            return_value = []
            for pollution_type in data:
                return_value.append(cls.from_dict(pollution_type))
            cls.stored = {i.id: i for i in return_value}

        return cls.stored


class Measurement(DDBobject):
    stored = {}

    def __init__(self, station, pollution_type, value, time, measurement_id=None):
        self.id = measurement_id or str(uuid.uuid4())
        self.station = station
        self.type = pollution_type
        self.value = value
        self.time = time

    @classmethod
    @coroutine
    def get(cls, measurement_id):
        if cls.stored.get(measurement_id, None):
            return cls.stored.get(measurement_id)

        data = yield DDBmeasurements().get(measurement_id)

        station = yield Station.get(data['station_id'])
        data['station'] = station
        del data['station_id']

        pollution_type = yield Type.get(data['type_id'])
        data['pollution_type'] = pollution_type
        del data['type_id']

        return cls.from_dict(data)

    @coroutine
    def save(self):
        result = yield DDBmeasurements().add(measurement=self)
        result = yield DDBlastmeasurements().add(measurement=self)
        return result

    @classmethod
    @coroutine
    def get_last(cls, station, pollution_type):
        data = yield DDBlastmeasurements().get(station, pollution_type)
        return_value = yield cls.get(measurement_id=data['measurement_id'])
        return return_value

    def __str__(self):
        return "{0} {1} {2} {3} {4}".format(self.id, self.station.id, self.type.id, self.value, self.time)
