import sys

import tornado.ioloop
import tornado.web
import models
from tornado import gen
import json


class MainHandler(tornado.web.RequestHandler):
    @gen.coroutine
    def get(self):
        stations = yield models.Station.get_all()
        pollutions = yield models.Type.get_all()
        last_measurements = yield models.Measurement.get_all_last()

        self.set_header('Content-Type', 'application/json')
        self.write(json.dumps(self.get_full_json_response(stations=stations, pollutions=pollutions,
                                                          last_measurements=last_measurements)))

    def get_full_json_response(self, stations, pollutions, last_measurements):
        result = dict()

        result['stations'] = [self.get_station_json_object(stations[station]) for station in stations]
        result['pollutions'] = [self.get_pollution_json_object(pollutions[pollution]) for pollution in pollutions]
        result['measurements'] = dict()
        for last_measurement in last_measurements:
            result['measurements'][last_measurement.id] = self.get_measurement_json_object(last_measurement)

        return result

    def get_station_json_object(self, station):
        return {
            "uuid": station.id,
            "name": station.name,
            "city": station.city,
            "longitude": station.longitude,
            "latitude": station.latitude
        }

    def get_pollution_json_object(self, pollution):
        return {
            "uuid": pollution.id,
            "shortname": pollution.shortname,
            "longname": pollution.longname,
            "description": pollution.description,
            "unit": pollution.unit,
            "norm": pollution.norm
        }

    def get_measurement_json_object(self, measurement):
        return {
            "uuid": measurement.id,
            "station": measurement.station.id,
            "type": measurement.type.id,
            "value": measurement.value,
            "time": measurement.time
        }


def make_app():
    return tornado.web.Application([
        (r"/view", MainHandler),
        (r'/()', tornado.web.StaticFileHandler, {'path': '/home/smok/smogonet/index.html'})
    ])

if __name__ == "__main__":
    app = make_app()
    port = 80
    if len(sys.argv) > 1:
        port = sys.argv[1]
    app.listen(port)
    tornado.ioloop.IOLoop.current().start()
