# -*- coding: utf-8 -*-
"""
This script is responsible for update measurement values.

"""
from uuid import UUID

import requests
import time
from tornado.ioloop import IOLoop
from dateutil.parser import parse
import models
from tornado.gen import coroutine

def get_station_by_id(station_id, station_data):
    """
    Returns station from dict of stations based on station_id

    :param station_id (int):
    :param station_data (dict):
    :return (Station):

    """
    station_id = str(UUID(int=int(station_id)))
    return station_data[station_id]


def get_type_by_name(type_name, type_data):
    """
    Returns pollution type from dict of types based on type_name

    :param type_name (str):
    :param type_data (dict):
    :return (Type):

    """
    for type in type_data.values():
        if type.shortname == type_name:
            return type


def get_timestamp_from_datetime(datetime_string):
    """
    Parses datetime value and return timestamp

    :param datetime_string (str):
    :return (int):

    """
    dt = parse(datetime_string)
    return time.mktime(dt.timetuple())


@coroutine
def main():
    """
    Main function

    :return (None):

    """
    apiURL = "http://powietrze.malopolska.pl/_powietrzeapi/api/dane?act=danemiasta&ci_id=1"

    r = requests.get(apiURL)
    actual_data = r.json()['dane']['actual']

    station_data = yield models.Station.get_all()
    type_data = yield models.Type.get_all()

    for station_measurements in actual_data:
        for measurement in station_measurements['details']:
            ready_station = get_station_by_id(measurement['st_id'], station_data)
            ready_type = get_type_by_name(measurement['par_name'], type_data)
            ready_measurement_value = measurement['o_value']
            ready_measurement_timestamp = get_timestamp_from_datetime(measurement['o_czas'])

            if ready_station and ready_type and ready_measurement_value and ready_measurement_timestamp:
                measurement = models.Measurement(station=ready_station, pollution_type=ready_type,
                                                 value=ready_measurement_value, time=ready_measurement_timestamp)
                yield measurement.save()

IOLoop.current().run_sync(main)
