import uuid

from db import DDBtypes, DDBmeasurements, DDBstations
from functools import partial
import hermes.backend.dict

from tornado.ioloop import IOLoop

cache = hermes.Hermes(hermes.backend.dict.Backend)

class DDBobject(object):
    @classmethod
    def from_dict(cls, data):
        return cls(**data)


class Station(DDBobject):
    stored = {}

    def __init__(self, api_station_id, city, longitude, latitude, name, station_id=None):
        self.id = station_id or str(uuid.uuid4())
        self.api_id = api_station_id
        self.city = city
        self.longitude = longitude
        self.latitude = latitude
        self.name = name

    @classmethod
    def get(cls, station_id):
        if cls.stored.get(station_id, None):
            return cls.stored.get(station_id)

        return cls.from_dict(IOLoop.instance().run_sync(partial(DDBstations().get, station_id)))

    def save(self):
        return IOLoop.instance().run_sync(partial(DDBstations().add, station=self))


class Type(DDBobject):
    stored = {}

    def __init__(self, shortname, unit, norm, longname, description, type_id=None):
        self.id = type_id or str(uuid.uuid4())
        self.shortname = shortname
        self.unit = unit
        self.norm = norm
        self.longname = longname
        self.description = description

    @classmethod
    def get(cls, type_id):
        if cls.stored.get(type_id, None):
            return cls.stored.get(type_id)

        return cls.from_dict(IOLoop.instance().run_sync(partial(DDBtypes().get, type_id)))

    def save(self):
        return IOLoop.instance().run_sync(partial(DDBtypes().add, pollution_type=self))


class Measurement(DDBobject):
    stored = {}

    def __init__(self, station, pollution_type, value, time, measurement_id=None):
        self.id = measurement_id or str(uuid.uuid4())
        self.station = station
        self.type = pollution_type
        self.value = value
        self.time = time

    @cache(ttl=6000)
    @classmethod
    def get(cls, measurement_id):
        if cls.stored.get(measurement_id, None):
            return cls.stored.get(measurement_id)

        data = IOLoop.instance().run_sync(partial(DDBmeasurements().get, measurement_id))

        data['station'] = Station.get(data['station_id'])
        del data['station_id']

        data['pollution_type'] = Type.get(data['type_id'])
        del data['type_id']

        return cls.from_dict(data)

    def save(self):
        return IOLoop.instance().run_sync(partial(DDBmeasurements().add, measurement=self))
