import uuid
from unittest.mock import MagicMock, patch

import time

from asynctest import CoroutineMock
from models import Type, Station, Measurement
from tornado.testing import AsyncTestCase, gen_test


class TypeTest(AsyncTestCase):
    def setUp(self):
        Type.stored = {}
        super(TypeTest, self).setUp()
        self.id = uuid.uuid4()
        self.shortname = "test_shortname"
        self.unit = "test_unit"
        self.norm = 57
        self.longname = "test_longname"
        self.description = "test_description"
        self.type = Type(shortname=self.shortname, unit=self.unit, norm=self.norm, longname=self.longname,
                         description=self.description, type_id=self.id)

    def tearDown(self):
        Type.stored = {}

    @gen_test
    def test_init(self):
        self.assertEqual(self.type.id, self.id)
        self.assertEqual(self.type.shortname, self.shortname)
        self.assertEqual(self.type.unit, self.unit)
        self.assertEqual(self.type.norm, self.norm)
        self.assertEqual(self.type.longname, self.longname)
        self.assertEqual(self.type.description, self.description)

    @gen_test
    def test_get_from_storage(self):
        Type.stored = {self.type.id: self.type}
        result = yield Type.get(self.type.id)

        self.assertEqual(result, self.type)

    @gen_test
    def test_get(self):
        with patch('models.DDBtypes') as mock_types:
            mock_types.return_value.get = CoroutineMock()
            mock_types.return_value.get.return_value = {
                'type_id': self.id,
                'shortname': self.shortname,
                'unit': self.unit,
                'norm': self.norm,
                'longname': self.longname,
                'description': self.description
            }
            result = yield Type.get(self.type.id)

        self.assertEqual(result.id, self.id)
        self.assertEqual(result.shortname, self.shortname)
        self.assertEqual(result.unit, self.unit)
        self.assertEqual(result.norm, self.norm)
        self.assertEqual(result.longname, self.longname)
        self.assertEqual(result.description, self.description)

    @gen_test
    def test_save(self):
        with patch('models.DDBtypes') as mock_type:
            mock_type.return_value.add = CoroutineMock()
            self.type.save()
            mock_type.return_value.add.assert_called_once_with(pollution_type=self.type)

    @gen_test
    def test_get_all(self):
        with patch('models.DDBtypes') as mock_type:
            mock_type.return_value.get_all = CoroutineMock()
            mock_type.return_value.get_all.return_value = [{
                'type_id': self.id,
                'shortname': self.shortname,
                'unit': self.unit,
                'norm': self.norm,
                'longname': self.longname,
                'description': self.description
            }]

            result = yield Type.get_all()

        expected = {self.type.id: self.type}

        self.assertCountEqual(result, expected)
        self.assertCountEqual(Type.stored, expected)


class StationTest(AsyncTestCase):
    def setUp(self):
        super(StationTest, self).setUp()
        Station.stored = {}
        self.id = uuid.uuid4()
        self.api_id = 1337
        self.city = "test_city"
        self.longitude = 12345
        self.latitude = 54321
        self.name = "test_name"
        self.station = Station(station_id=self.id, api_station_id=self.api_id, city=self.city, longitude=self.longitude,
                               latitude=self.latitude, name=self.name)

    def tearDown(self):
        Station.stored = {}

    @gen_test
    def test_init(self):
        self.assertEqual(self.station.id, self.id)
        self.assertEqual(self.station.api_id, self.api_id)
        self.assertEqual(self.station.city, self.city)
        self.assertEqual(self.station.longitude, self.longitude)
        self.assertEqual(self.station.latitude, self.latitude)
        self.assertEqual(self.station.name, self.name)

    @gen_test
    def test_get_from_storage(self):
        Station.stored = {self.station.id: self.station}
        result = yield Station.get(self.station.id)

        self.assertEqual(result, self.station)

    @gen_test
    def test_get(self):
        with patch('models.DDBstations') as mock_types:
            mock_types.return_value.get = CoroutineMock()
            mock_types.return_value.get.return_value = {
                'station_id': self.id,
                'api_station_id': self.api_id,
                'city': self.city,
                'longitude': self.longitude,
                'latitude': self.latitude,
                'name': self.name
            }
            result = yield Station.get(self.station.id)

        self.assertEqual(result.id, self.id)
        self.assertEqual(result.api_id, self.api_id)
        self.assertEqual(result.city, self.city)
        self.assertEqual(result.longitude, self.longitude)
        self.assertEqual(result.latitude, self.latitude)
        self.assertEqual(result.name, self.name)

    @gen_test
    def test_save(self):
        with patch('models.DDBstations') as mock_station:
            mock_station.return_value.add = CoroutineMock()
            self.station.save()
            mock_station.return_value.add.assert_called_once_with(station=self.station)

    @gen_test
    def test_get_all(self):
        with patch('models.DDBstations') as mock_station:
            mock_station.return_value.get_all = CoroutineMock()
            mock_station.return_value.get_all.return_value = [{
                'station_id': self.id,
                'api_station_id': self.api_id,
                'city': self.city,
                'longitude': self.longitude,
                'latitude': self.latitude,
                'name': self.name
            }]

            result = yield Station.get_all()

        expected = {self.station.id: self.station}

        self.assertCountEqual(result, expected)
        self.assertCountEqual(Station.stored, expected)


class MeasurementTest(AsyncTestCase):
    def setUp(self):
        super(MeasurementTest, self).setUp()
        Measurement.stored = {}
        self.id = uuid.uuid4()
        self.station = MagicMock()
        self.type = MagicMock()
        self.value = 1337
        self.time = time.time()
        self.meas = Measurement(measurement_id=self.id, station=self.station, pollution_type=self.type,
                                value=self.value, time=self.time)

    def tearDown(self):
        Measurement.stored = {}

    @gen_test
    def test_init(self):
        self.assertEqual(self.meas.id, self.id)
        self.assertEqual(self.meas.station, self.station)
        self.assertEqual(self.meas.type, self.type)
        self.assertEqual(self.meas.value, self.value)
        self.assertEqual(self.meas.time, self.time)

    @gen_test
    def test_get_from_storage(self):
        Measurement.stored = {self.meas.id: self.meas}
        result = yield Measurement.get(self.meas.id)

        self.assertEqual(result, self.meas)

    @gen_test
    def test_get(self):
        with patch('models.Station') as mock_station:
            mock_station.get = CoroutineMock()
            mock_station.get.return_value = self.station

            with patch('models.Type') as mock_type:
                mock_type.get = CoroutineMock()
                mock_type.get.return_value = self.type

                with patch('models.DDBmeasurements') as mock_types:
                    mock_types.return_value.get = CoroutineMock()
                    mock_types.return_value.get.return_value = {
                        'measurement_id': self.id,
                        'station_id': self.station.id,
                        'type_id': self.type.id,
                        'value': self.value,
                        'time': self.time,
                    }
                    result = yield Measurement.get(self.station.id)

                mock_type.get.assert_called_once_with(self.type.id)
            mock_station.get.assert_called_once_with(self.station.id)

        self.assertEqual(result.id, self.id)
        self.assertEqual(result.station, self.station)
        self.assertEqual(result.type, self.type)
        self.assertEqual(result.value, self.value)
        self.assertEqual(result.time, self.time)

    @gen_test
    def test_save(self):
        with patch('models.DDBmeasurements') as mock_measurements:
            with patch('models.DDBlastmeasurements') as mock_last:
                mock_measurements.return_value.add = CoroutineMock()
                mock_last.return_value.add = CoroutineMock()

                self.meas.save()
                mock_measurements.return_value.add.assert_called_once_with(measurement=self.meas)
                mock_last.return_value.add.assert_called_once_with(measurement=self.meas)

    @gen_test
    def test_get_last(self):
        Measurement.stored = {self.meas.id: self.meas}
        with patch('models.DDBlastmeasurements') as mock_last:
            mock_last.return_value.get = CoroutineMock()
            mock_last.return_value.get.return_value = {
                'measurement_id': self.meas.id
            }

            result = yield Measurement.get_last(station=self.station, pollution_type=self.type)
            mock_last.return_value.get.assert_called_once_with(self.station, self.type)

        self.assertEqual(result, self.meas)

    @gen_test
    def test_get_all_last(self):
        Measurement.stored = {self.meas.id: self.meas}
        with patch('models.DDBlastmeasurements') as mock_last:
            mock_last.return_value.get_all = CoroutineMock()
            mock_last.return_value.get_all.return_value = [{
                'measurement_id': self.meas.id
            }]

            result = yield Measurement.get_all_last()
            mock_last.return_value.get_all.assert_called_once_with()

        self.assertCountEqual(result, [self.meas])

    def test_str(self):
        expected = "{0} {1} {2} {3} {4}".format(self.id, self.station.id, self.type.id, self.value, self.time)
        result = str(self.meas)

        self.assertEqual(result, expected)