import uuid

from db import DDBtypes, DDBmeasurements, DDBstations
from functools import partial
from tornado.gen import coroutine

from tornado.ioloop import IOLoop


class DDBobject(object):
    @classmethod
    @coroutine
    def from_dict(cls, data):
        return cls(**data)


class Station(DDBobject):
    stored = {}

    def __init__(self, api_station_id, city, longitude, latitude, name, station_id=None):
        self.id = station_id or str(uuid.uuid4())
        self.api_id = api_station_id
        self.city = city
        self.longitude = longitude
        self.latitude = latitude
        self.name = name

    @classmethod
    @coroutine
    def get(cls, station_id):
        if cls.stored.get(station_id, None):
            return cls.stored.get(station_id)

        result = yield DDBstations().get(station_id)
        return cls.from_dict(result)

    @coroutine
    def save(self):
        result = yield DDBstations().add(station=self)
        return result


class Type(DDBobject):
    stored = {}

    def __init__(self, shortname, unit, norm, longname, description, type_id=None):
        self.id = type_id or str(uuid.uuid4())
        self.shortname = shortname
        self.unit = unit
        self.norm = norm
        self.longname = longname
        self.description = description

    @classmethod
    @coroutine
    def get(cls, type_id):
        if cls.stored.get(type_id, None):
            return cls.stored.get(type_id)

        result = yield DDBtypes().get(type_id)
        return cls.from_dict(result)

    def save(self):
        result = yield (DDBtypes().add(pollution_type=self))
        return result


class Measurement(DDBobject):
    stored = {}

    def __init__(self, station, pollution_type, value, time, measurement_id=None):
        self.id = measurement_id or str(uuid.uuid4())
        self.station = station
        self.type = pollution_type
        self.value = value
        self.time = time

    @classmethod
    @coroutine
    def get(cls, measurement_id):
        if cls.stored.get(measurement_id, None):
            return cls.stored.get(measurement_id)

        data = yield DDBmeasurements().get(measurement_id)

        station = yield Station.get(data['station_id'])
        data['station'] = station.result()
        del data['station_id']

        pollution_type = yield Type.get(data['type_id'])
        data['pollution_type'] = pollution_type.result()
        del data['type_id']

        return cls.from_dict(data)

    @coroutine
    def save(self):
        result = yield DDBmeasurements().add(measurement=self)
        return result
